/* eslint-disable flowtype/require-return-type, flowtype/require-parameter-type */

import test from 'ava';
import coscript from 'coscript';
import fs from 'fs';
import path from 'path';
import lodash from 'lodash';

const LIBRARY_NAME = 'll-test';
const LIBRARY_FOLDER = '/tmp';
const LIBRARY_PATH = path.join(LIBRARY_FOLDER, `${LIBRARY_NAME}.library`);
const METADATA_PATH = path.join(LIBRARY_PATH, 'metadata.json');

const SUBMIT_BUTTON_SYMBOL_ID = '1CA4F273-663F-4A71-AC81-09DF50091A59';
const WARNING_BUTTON_SYMBOL_ID = '00ABA800-E14F-4B2D-8DA3-779C168811AA';

const SUBMIT_BUTTON_ID = '9A17AD01-1B31-4AF8-809C-E5121D8B1108';
const WARNING_BUTTON_ID = 'B6E60363-CF16-4A98-A41B-4097F7A076D3';

const TEXT_STYLE_LAYER_1_ID = '99158DBC-90D1-4894-BF39-E315E24529CF';
const TEXT_STYLE_1_ID = '0CAF8A71-8826-4543-B91F-4995A42F43A6';

const LAYER_FOR_TEXT_STYLE_ID = '1F91BAB0-510C-4520-B66C-5BE04B4F5BFF';
const LAYER_WITHOUT_TEXT_STYLE_ID = '99D33663-E4F1-40DF-81BE-025D7219065B';

const TEXT_STYLE_1_JSON = fs.readFileSync('./test/fixtures/text-style-1.json', 'utf8');
const OLD_LIBRARY_PATH = './test/fixtures/rock.library';

let metadata;

function runCommand(command, args, callback) {
  const options = {
    command,
    args
  };
  coscript.runFileWithArgs('test/coscript/run-command.js.coscript', options, 'Sketch', callback);
}

function runCoscript(name, args, callback) {
  if (typeof args === 'function') {
    callback = args;
    args = undefined;
  }

  if (args) {
    coscript.runFileWithArgs(`test/coscript/${name}.js.coscript`, args, 'Sketch', callback);
  } else {
    coscript.runFile(`test/coscript/${name}.js.coscript`, 'Sketch', callback);
  }
}

function selectLayersOnSymbolsPage(ids, callback) {
  runCoscript('select-layers-on-symbols-page', { ids }, callback);
}

function selectLayers(ids, callback) {
  runCoscript('select-layers', { ids }, callback);
}

test.serial.only.cb('create library', (t) => {
  const args = {
    path: LIBRARY_FOLDER,
    name: LIBRARY_NAME
  };

  runCommand('create-library', args, (err) => {
    t.falsy(err);
    fs.readFile(METADATA_PATH, 'utf8', (err, jsonStr) => {
      t.falsy(err);
      const data = JSON.parse(jsonStr);
      metadata = data;
      t.is(data.name, 'll-test');
      t.is(data.categories.length, 3);
      t.is(data.categories[0].name, 'Uncategorized');
      t.is(data.categories[1].name, 'Colors');
      t.is(data.categories[2].name, 'Text Styles');
      t.end();
    });
  });
});

test.serial.cb('import old library' , (t) => {
  const args = {
    path: OLD_LIBRARY_PATH
  };

  runCommand('import-library', args, (err) => {
    t.falsy(err);
    t.end();
  });
});

function readJson(filePath) {
  return JSON.parse(fs.readFileSync(filePath, 'utf8'));
}

function assertTextStyle(t, textStyleId, expectedMetadata) {
  const textStylePath = path.join(LIBRARY_PATH, `${textStyleId}.textstyle`);
  const previewPath = path.join(textStylePath, 'preview.png');
  const metadataPath = path.join(textStylePath, 'metadata.json');
  const dataPath = path.join(textStylePath, 'data');

  t.truthy(fs.statSync(textStylePath).isDirectory());
  t.truthy(fs.statSync(previewPath).isFile());
  t.not(fs.statSync(previewPath).size, 0);
  t.truthy(fs.statSync(dataPath).isFile());

  t.deepEqual(readJson(metadataPath), expectedMetadata);
}

function assertLayer(t, layerId, expectedMetadata) {
  const layerPath = path.join(LIBRARY_PATH, `${layerId}.layer`);
  const previewPath = path.join(layerPath, 'preview.png');
  const metadataPath = path.join(layerPath, 'metadata.json');
  const symbolsPath = path.join(LIBRARY_PATH, 'symbols');

  t.truthy(fs.statSync(layerPath).isDirectory());
  t.truthy(fs.statSync(previewPath).isFile());
  t.not(fs.statSync(previewPath).size, 0);

  t.deepEqual(readJson(metadataPath), expectedMetadata);

  expectedMetadata.symbols.forEach((symbolId) => {
    const symbolPath = path.join(symbolsPath, `${symbolId}.symbol`);
    t.truthy(fs.statSync(symbolPath).isFile());
    t.not(fs.statSync(symbolPath).size, 0);
  });
}

test.serial.cb('add layers', (t) => {
  const args = {
    libraryPath: LIBRARY_PATH,
    parentId: metadata.categories[0].id,
    index: 0
  };

  const submitButtonExpectedMetadata = {
    parentId: args.parentId,
    version: 1,
    symbols: [
      '1CA4F273-663F-4A71-AC81-09DF50091A59'
    ],
    name: 'Submit Button',
    index: 0
  };

  const warningButtonExpectedMetadata = {
    parentId: args.parentId,
    version: 1,
    symbols: [
      '00ABA800-E14F-4B2D-8DA3-779C168811AA'
    ],
    name: 'Warning Button',
    index: 1
  };

  selectLayersOnSymbolsPage([SUBMIT_BUTTON_ID, WARNING_BUTTON_ID], (err) => {
    t.falsy(err);

    runCommand('add-layers', args, (err) => {
      t.falsy(err);
      assertLayer(t, SUBMIT_BUTTON_SYMBOL_ID, submitButtonExpectedMetadata);
      assertLayer(t, WARNING_BUTTON_SYMBOL_ID, warningButtonExpectedMetadata);
      t.end();
    });
  });
});

test.serial.only.cb('add text styles', (t) => {
  const args = {
    libraryPath: LIBRARY_PATH,
    parentId: metadata.categories[2].id,
    index: 0
  };

  const textStyle1ExpectedMetadata = {
    index: 0,
    name: 'Type something Style',
    parentId: args.parentId
  };

  selectLayers([TEXT_STYLE_LAYER_1_ID], (err) => {
    t.falsy(err);

    runCommand('add-text-styles', args, (err) => {
      t.falsy(err);
      assertTextStyle(t, TEXT_STYLE_1_ID, textStyle1ExpectedMetadata);
      t.end();
    });
  });
});

test.serial.cb('apply text style', (t) => {
  const args = {
    libraryPath: LIBRARY_PATH,
    objectID: TEXT_STYLE_1_ID
  };

  selectLayers([LAYER_FOR_TEXT_STYLE_ID], (err) => {
    t.falsy(err);

    runCommand('set-text-style', args, (err) => {
      t.falsy(err);

      runCoscript('get-text-style', (err, data) => {
        t.falsy(err);

        t.true(data.trim() === TEXT_STYLE_1_JSON.trim());

        t.end();
      });
    });
  });
});

function getTextStyles() {
  return fs.readdirSync(LIBRARY_PATH).filter((fileName) => {
    return path.extname(fileName) === '.textstyle';
  });
}

test.serial.cb('create text style from layer without text style', (t) => {
  const args = {
    libraryPath: LIBRARY_PATH,
    parentId: metadata.categories[2].id,
    index: 1
  };

  const currentTextStyles = getTextStyles();

  const expectedMetadata = {
    index: 1,
    name: 'Layer without Text Style Style',
    parentId: args.parentId
  };

  selectLayers([LAYER_WITHOUT_TEXT_STYLE_ID], (err) => {
    t.falsy(err);

    runCommand('add-text-styles', args, (err) => {
      t.falsy(err);

      const addedTextStyles = lodash.difference(getTextStyles(), currentTextStyles);
      t.is(addedTextStyles.length, 1);

      const addedTextStyle = addedTextStyles[0];
      const addedTextStyleId = path.basename(addedTextStyle, path.extname(addedTextStyle));

      assertTextStyle(t, addedTextStyleId, expectedMetadata);
      t.end();
    });
  });
});
